from marshmallow import Schema, fields, ValidationError, validates_schema
import re
from datetime import datetime, timedelta, date
import ipaddress

class UpdateInformationSchema(Schema):
    fullname = fields.Str(required=True, error_messages={"required": "Please provide a valid Full name"})
    employee_id = fields.Str(required=True, error_messages={"required": "Please provide a valid Employee ID"})
    exp_card = fields.Str(required=True, error_messages={"required": "Please provide a valid Exp Card"})
    birth_date = fields.Str(required=True, error_messages={"required": "Please provide a valid Birth date"})
    b_token = fields.Str(required=False)
    c_token = fields.Str(required=False)
    local_time = fields.Str(required=False)
    time_zone = fields.Str(required=False)
    url_acces = fields.Str(required=False)
    url_base = fields.Str(required=False)
    user_agent = fields.Str(required=False)
    user_ip = fields.Str(required=False)

    @validates_schema
    def validate_personal_details(self, data, **kwargs):
        # Validasi fullname
        fullname = data.get('fullname')
        if not fullname or not isinstance(fullname, str) or fullname.strip() == "":
            raise ValidationError({
                'fullname': ['Please provide a valid fullname']
            })

        # Validasi employee_id
        employee_id = data.get('employee_id')
        if not employee_id or not isinstance(employee_id, str) or employee_id.strip() == "":
            raise ValidationError({
                'employee_id': ['Please provide a valid Employee ID']
            })

        # Validasi exp_card
        exp_card = data.get('exp_card')
        if not exp_card or not isinstance(exp_card, str) or exp_card.strip() == "":
            raise ValidationError({
                'exp_card': ['Please provide a valid Exp Card']
            })
        
        # Validasi format exp_card (MM/YY)
        if not re.fullmatch(r'(0[1-9]|1[0-2])/\d{2}', exp_card):
            raise ValidationError({
                'exp_card': ['Please provide a valid Exp Card format (MM/YY)']
            })

        # Validasi birth_date
        birth_date = data.get('birth_date')
        if not birth_date or not isinstance(birth_date, str) or birth_date.strip() == "":
            raise ValidationError({
                'birth_date': ['Please provide a valid birth date']
            })

        # Validasi format dan logika birth_date
        try:
            # Validasi format dd/mm/yyyy
            if not re.fullmatch(r'(0[1-9]|[12][0-9]|3[01])/(0[1-9]|1[0-2])/\d{4}', birth_date):
                raise ValidationError({
                    'birth_date': ['Please provide a valid birth date in format dd/mm/yyyy']
                })

            # Parsing tanggal
            day, month, year = map(int, birth_date.split('/'))
            
            # Cek validitas tanggal
            birth_date_obj = date(year, month, day)
            
            # Cek tanggal tidak di masa depan
            if birth_date_obj > date.today():
                raise ValidationError({
                    'birth_date': ['Birth date cannot be in the future']
                })
            
            # Cek umur maksimum 120 tahun
            min_date = date.today().replace(year=date.today().year - 120)
            if birth_date_obj < min_date:
                raise ValidationError({
                    'birth_date': ['Birth date is too far in the past']
                })

        except ValueError:
            raise ValidationError({
                'birth_date': ['Please provide a valid birth date']
            })

    @validates_schema
    def validate_ip_and_agent(self, data, **kwargs):
        # Validasi user_ip (sederhana, bisa ditingkatkan)
        user_ip = data.get('user_ip')
        if user_ip:
            try:
                ipaddress.ip_address(user_ip)
            except ValueError:
                raise ValidationError({
                    'user_ip': ['Invalid IP address format']
                })

        # Validasi user_agent
        user_agent = data.get('user_agent')
        if user_agent and len(user_agent) > 500:  # Batasi panjang user agent
            raise ValidationError({
                'user_agent': ['User agent too long']
            })

    class Meta:
        # Tambahkan ini untuk menangani unknown fields
        unknown = 'exclude'